﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data;


//access resource.resx
using System.Resources;

//our data access class library
using DataAccess;

/// <summary>
/// contains methods and properties to retrieve and save patient data
/// </summary>
public class CPatientDataXfer : CData
{
    /// <summary>
    /// US:5729 US:5349 constructor
    /// </summary>
    /// <param name="data"></param>
    public CPatientDataXfer(CData data)
        : base(data)
    {
        //constructors are not inherited in c#!
    }

    public CStatus GetPatientIDDS(string strDBConnString,
                                   bool bAudit,
                                   long lUserID,
                                   string strPatientID,
                                   out DataSet ds)
    {
        //initialize parameters
        ds = null;

        CStatus status = new CStatus();

        //get a connection
        CDataConnection conn = new CDataConnection();
        //status = conn.Connect(strDBConnString, bAudit, this);
        //if (!status.Status)
        //{
         //   return status;
        //}

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(base.SessionID,
                                                          base.ClientIP,
                                                          base.UserID);

        pList.AddInputParameter("pi_vPatientID", strPatientID);

        //get the dataset
        CDataSet cds = new CDataSet();
        status = cds.GetOracleDataSet(base.DBConn,
                                       "PCK_REVAMP_TIU.GetMDWSPatientIDRS",
                                       pList,
                                       out ds);

        conn.Close();

        return status;
    }

    /// <summary>
    /// US:5729 US:5349 saves a loaded patient to the database.
    /// </summary>
    /// <param name="pdi"></param>
    /// <returns></returns>
    public CStatus SavePatient( long lXferSystemID,
                                string strKey,
                                System.Web.SessionState.HttpSessionState WebSession,
                                CPatientDataItemXfer pdi,
                                out string strPatientID)
    {
        //our patient id
        strPatientID = "";

        //check the patient id
        if (pdi == null || String.IsNullOrEmpty(pdi.PatientID))
        {
            return new CStatus(false, k_STATUS_CODE.Failed, "Could not save patient, invalid data!"); ;
        }

        //check the region/site
        long lRegionID = this.RegionID;
        long lSiteID = this.SiteID;
        if (lRegionID < 1 || lSiteID < 1)
        {
            return new CStatus(false, k_STATUS_CODE.Failed, "Could not save patient, invalid region/site data!"); ;
        }

        //check the data connection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        
        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_nXferSystemID", lXferSystemID);
        pList.AddInputParameter("pi_vKey", strKey);

        pList.AddInputParameter("pi_nRegionID", lRegionID);
        pList.AddInputParameter("pi_nSiteID", lSiteID);

        pList.AddInputParameter("pi_vMDWSPatientID", pdi.PatientID);
        pList.AddInputParameter("pi_vSSN", pdi.SSN);
        pList.AddInputParameter("pi_dtDOB", pdi.DOB);
        pList.AddInputParameter("pi_vFirstName", pdi.FirstName);
        pList.AddInputParameter("pi_vFullName", pdi.FullName);
        pList.AddInputParameter("pi_vLastName", pdi.LastName);
        pList.AddInputParameter("pi_vMI", pdi.MI);
        pList.AddInputParameter("pi_nSex", (long)pdi.Sex);

        pList.AddInputParameter("pi_vHomeAddr1", pdi.HomeAddr1);
        pList.AddInputParameter("pi_vHomeAddr2", pdi.HomeAddr2);
        pList.AddInputParameter("pi_vHomeCity", pdi.HomeCity);
        pList.AddInputParameter("pi_vHomeState", pdi.HomeState);
        pList.AddInputParameter("pi_vHomeZip", pdi.HomeZipCode);

        pList.AddInputParameter("pi_vHomePhone", pdi.HomePhone);
        pList.AddOutputParameter("po_vPatientID", strPatientID);

        //execute the transfer SP
        CStatus statusXfer = DBConn.ExecuteOracleSP("PCK_XFER.XferPatient", pList);
        if (statusXfer.Status)
        {
            CDataParameter pPatID = pList.GetItemByName("po_vPatientID");
            if (pPatID != null)
            {
                strPatientID = pPatID.StringParameterValue;
            }
        }

        return statusXfer;
    }


    /// <summary>
    /// builds the patient blurb for display
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="strBlurb"></param>
    /// <returns></returns>
    public CStatus GetPatientBlurb(string strPatientID, out string strBlurb)
    {
        strBlurb = string.Empty;
        CPatientDataXfer p = new CPatientDataXfer(this);
        CPatientDataItemXfer di = null;
        CStatus status = p.GetPatientDI(strPatientID, out di);
        if (status.Status)
        {
            strBlurb = di.FirstName + " " + di.LastName + " " + di.SSNLast4 + ", " + di.Age + " yo " + di.SexLabel;
        }

        return status;
    }

    /// <summary>
    /// saves a loaded patient to the database.
    /// pass in lUserID > 0 if you want this patient to
    /// belong to this provider
    /// </summary>
    /// <param name="pdi"></param>
    /// <returns></returns>
    public CStatus SavePatient(
        long lProviderUserID,
        long lTeamID,
        long lSpecialtyID,
        long lWardID,
        long lClinicID,
        long lXferSystemID,
        CPatientDataItemXfer pdi)
    {
        if (pdi == null || String.IsNullOrEmpty(pdi.PatientID))
        {
            return new CStatus(false, k_STATUS_CODE.Failed, "Could not save patient, invalid data!"); ;
        }

        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_nXferSystemID", lXferSystemID);
        pList.AddInputParameter("pi_vPatientID", pdi.PatientID);
        pList.AddInputParameter("pi_nProviderUserID", lProviderUserID);
        pList.AddInputParameter("pi_nTeamID", lTeamID);
        pList.AddInputParameter("pi_nSpecialtyID", lSpecialtyID);
        pList.AddInputParameter("pi_nWardID", lWardID);
        pList.AddInputParameter("pi_nClinicID", lClinicID);
        pList.AddInputParameter("pi_vSSN", pdi.SSN);
        pList.AddInputParameter("pi_dtDOB", pdi.DOB);
        pList.AddInputParameter("pi_vFirstName", pdi.FirstName);
        pList.AddInputParameter("pi_vFullName", pdi.FullName);
        pList.AddInputParameter("pi_vLastName", pdi.LastName);
        pList.AddInputParameter("pi_vMI", pdi.MI);
        pList.AddInputParameter("pi_nSex", (long)pdi.Sex);

        return DBConn.ExecuteOracleSP("PCK_PATIENT.SavePatient", pList);
    }


    /// <summary>
    /// method
    /// US:838
    /// retrieves a dataset of patients matching search criteria
    /// </summary>
    /// <param name="dtEventStartDate"></param>
    /// <param name="dtEventEndDate"></param>
    /// <param name="strLastName"></param>
    /// <param name="strLSSN"></param>
    /// <param name="lChecklistID"></param>
    /// <param name="lChecklistStatusID"></param>
    /// <param name="strTeamIDs"></param>
    /// <param name="strWardIDs"></param>
    /// <param name="strSpecialtyIDs"></param>
    /// <param name="strClinicIDs"></param>
    /// <param name="ds"></param>
    /// <param name="lStatusCode"></param>
    /// <param name="strStatus"></param>
    /// <returns></returns>
    public CStatus GetPatientSearchDS(
        DateTime dtEventStartDate,
        DateTime dtEventEndDate,
        string strLastName,
        string strLast4,
        long lChecklistID,
        long lChecklistStatusID,
        string strUsrID,
        string strTeamID,
        string strWardID,
        string strSpecialtyID,
        string strClinicID,
        long lServiceID, 
        out DataSet ds)
    {
        //initialize parameters
        ds = null;

        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //move patients from MDWS if needed
        if (MDWSTransfer)
        {
            string strMatch = String.Empty;
            if (!String.IsNullOrEmpty(strLast4))
            {
                strMatch = strLast4;
            }
            else
            {
                if (!String.IsNullOrEmpty(strLastName))
                {
                    strMatch = strLastName;
                }
            }

           // if (!String.IsNullOrEmpty(strMatch))
           // {
           //     long lCount = 0;
           //     CMDWSOps ops = new CMDWSOps(this);
          //      ops.GetMDWSMatchPatients(strMatch, out lCount);
           // }
        }


        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_dtEventStartDate", dtEventStartDate);
        pList.AddInputParameter("pi_dtEventEndDate", dtEventEndDate);
        pList.AddInputParameter("pi_vLastName", strLastName);
        pList.AddInputParameter("pi_vLSSN", strLast4);
        pList.AddInputParameter("pi_nChecklistID", lChecklistID);
        pList.AddInputParameter("pi_nChecklistStatusID", lChecklistStatusID);
        pList.AddInputParameter("pi_vUsrID", strUsrID);
        pList.AddInputParameter("pi_vTeamID", strTeamID);
        pList.AddInputParameter("pi_vWardID", strWardID);
        pList.AddInputParameter("pi_vSpecialtyID", strSpecialtyID);
        pList.AddInputParameter("pi_vClinicID", strClinicID);
        pList.AddInputParameter("pi_nServiceID", lServiceID);
        
        //get the dataset
        CDataSet cds = new CDataSet();
        return cds.GetOracleDataSet(
            DBConn,
            "PCK_PATIENT.GetPatientSearchRS",
            pList,
            out ds);
    }


    /// <summary>
    /// method
    /// retrieves a dataset of patients matching the search criteria
    /// </summary>
    /// <param name="dtEventStartDate"></param>
    /// <param name="dtEventEndDate"></param>
    /// <param name="lChecklistID"></param>
    /// <param name="lChecklistStatusID"></param>
    /// <param name="ds"></param>
    /// <returns></returns>
    public CStatus GetMultiPatientSearchDS(
        DateTime dtEventStartDate,
        DateTime dtEventEndDate,
        long lChecklistID,
        long lChecklistStatusID,
        long lServiceID,
        out DataSet ds)
    {
        //initialize parameters
        ds = null;

        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_dtEventStartDate", dtEventStartDate);
        pList.AddInputParameter("pi_dtEventEndDate", dtEventEndDate);
        pList.AddInputParameter("pi_nChecklistID", lChecklistID);
        pList.AddInputParameter("pi_nChecklistStatusID", lChecklistStatusID);
        pList.AddInputParameter("pi_nServiceID", lServiceID);

        //get the dataset
        CDataSet cds = new CDataSet();
        return cds.GetOracleDataSet(
            base.DBConn,
            "PCK_PATIENT.GetMultiPatientSearchRS",
            pList,
            out ds);
    }

    /// <summary>
    /// method
    /// retrieves a dataset of patient checklist ids matching the search criteria
    /// </summary>
    /// <param name="dtEventStartDate"></param>
    /// <param name="dtEventEndDate"></param>
    /// <param name="lChecklistID"></param>
    /// <param name="lChecklistStatusID"></param>
    /// <param name="ds"></param>
    /// <returns></returns>
    public CStatus GetMultiPatientPatCLIDSearchDS(
        DateTime dtEventStartDate,
        DateTime dtEventEndDate,
        long lChecklistID,
        long lChecklistStatusID,
        long lServiceID,
        out DataSet ds)
    {
        //initialize parameters
        ds = null;

        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_dtEventStartDate", dtEventStartDate);
        pList.AddInputParameter("pi_dtEventEndDate", dtEventEndDate);
        pList.AddInputParameter("pi_nChecklistID", lChecklistID);
        pList.AddInputParameter("pi_nChecklistStatusID", lChecklistStatusID);
        pList.AddInputParameter("pi_nServiceID", lServiceID);
        
        //get the dataset
        CDataSet cds = new CDataSet();
        return cds.GetOracleDataSet(
            base.DBConn,
            "PCK_PATIENT.GetMultiPatientPatCLIDSearchRS",
            pList,
            out ds);
    }

    /// <summary>
    /// get user patient ds
    /// </summary>
    /// <param name="lUserID"></param>
    /// <param name="ds"></param>
    /// <returns></returns>
    public CStatus GetUserPatientDS(long lUserID, out DataSet ds)
    {
        //initialize parameters
        ds = null;

        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_nProviderUserID", lUserID);

        //get the dataset
        CDataSet cds = new CDataSet();
        return cds.GetOracleDataSet(
            DBConn,
            "PCK_PATIENT.GetUserPatientsRS",
            pList,
            out ds);
    }

    /// <summary>
    /// loads a patient data item by patient id
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="itm"></param>
    /// <param name="lStatusCode"></param>
    /// <param name="strStatus"></param>
    /// <returns></returns>
    public CStatus GetPatientDI(string strPatientID, out CPatientDataItemXfer itm)
    {
        //initialize parameters
        itm = null;

        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_vPatientID", strPatientID);

        //get the dataset
        DataSet ds = null;
        CDataSet cds = new CDataSet();
        status = cds.GetOracleDataSet(
            base.DBConn,
            "PCK_PATIENT.GetPatientIDRS",
            pList,
            out ds);

        if (!status.Status)
        {
            return status;
        }

        itm = new CPatientDataItemXfer(ds);

        return status;
    }

    /// <summary>
    /// returns a dataset of patient item components
    /// filtered by patient checklist id (PAT_CL_ID)
    /// <param name="lPatCLID"></param>
    /// <param name="ds"></param>
    /// <param name="lStatusCode"></param>
    /// <param name="strStatus"></param>
    /// <returns></returns>
    public CStatus GetPatItemCompsByPatCLIDDS(long lPatCLID, out DataSet ds)
    {
        //initialize parameters
        ds = null;

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_nPatCLID", lPatCLID);

        //get the dataset
        CDataSet cds = new CDataSet();
        return cds.GetOracleDataSet(
            base.DBConn,
            "PCK_PATIENT.GetPatItemCompsByPatCLIDRS",
            pList,
            out ds);
    }

    /// <summary>
    /// returns a dataset of patient item components
    /// filtered by patient checklist id (PAT_CL_ID)
    public CStatus GetPatItemsByPatCLIDDS(long lPatCLID, out DataSet ds)
    {
        //initialize parameters
        ds = null;

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_nPatCLID", lPatCLID);

        //get the dataset
        CDataSet cds = new CDataSet();
        return cds.GetOracleDataSet(
            base.DBConn,
            "PCK_PATIENT.GetPatItemsByPatCLIDRS",
            pList,
            out ds);
    }

    /// <summary>
    /// returns the item component state base on the primary key
    /// </summary>
    /// <param name="lPatItemID"></param>
    /// <param name="lItemCompID"></param>
    /// <param name="lItemCompState"></param>
    /// <returns></returns>
    public CStatus GetItemCompStateByPKey(
        long lPatItemID,
        long lItemCompID,
        out long lItemCompState)
    {
        //initialize parameters
        lItemCompState = 0;

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(SessionID, ClientIP, UserID);

        pList.AddInputParameter("pi_nPatItemID", lPatItemID);
        pList.AddInputParameter("pi_nItemCompID", lItemCompID);
        pList.AddOutputParameter("po_nItemCompState", lItemCompState);

        //get the dataset
        DataSet ds = null;
        CDataSet cds = new CDataSet();
        status = cds.GetOracleDataSet(
            base.DBConn,
            "PCK_PATIENT.GetItemCompStateByPKey",
            pList,
            out ds);

        if (!status.Status)
        {
            return status;
        }

        //get the Item Component State returned from the SP call
        lItemCompState = pList.GetParamLongValue("po_nItemCompState");

        return status;
    }
}
